#pragma rtGlobals=1		// Use modern global access method.
#include <DSP Window Functions>				//these need to be included for the BigPSD function
#pragma ModuleName=xARFitFuncs
#pragma IgorVersion=6.12		//  FitFunc/NWOK

////////////////////////////////////////////////////////////////////
//Functions to assist in analyzing two-state switching data
////////////////////////////////////////////////////////////////////
//Landon Prisbrey
//(landonprisbrey@gmail.com)
//Minot Nanoelectronics Group
//Department of Physics
//Oregon State University
////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////
//Quick overview of functions
////////////////////////////////////////////////////////////////////

//////////////COLOR CODING
//MakeColors("wavein")			Prepares for color coding data in 'wavein'. (Use before the function 'colors')
//Color("wavein", "red")			Colors section of 'wavein' specified by the cursor positions a certain color (in this case red)
//Stitch("wavein")					Deletes 'NaN' values, and 'stitches' together the numerical part of a wave. Useful following color coding in order to do further analysis.
//DeleteMe(wavein)				Sets a section of a wave to NaN (as specified by the cursor positions)

//////////////ANALYSIS
//CorrelateMe("wavein")			Normalizes & autocorrelates

//////////////FOURIER ANALYSIS
//MakeAudio(wavein)				Creates an audio wave for ASCII data. Next step is to save the data as an audio file (Data -> Save Waves -> Save Wav sound file)
//PSDme("wavein")				Performs PSD and plots in a logical way - should be easy to add fitting later
//FilterMe("wavein", time, sharp)	Removes timescales longer than 'time' from 'wavein' using a low pass filter. 'sharp' is the number of fourier coeficients to use - use odd numbers (the larger the number, the sharper drop at the cut off time - good starting value = 1001). The difference is then plotted. Handy tool for looking for two state switching.

//////////////PLOTTING
//PlotMe(wavein)					Plots data in my favorite way
//Lines()							Changes top plot to lines
//Linesanddots()					Changes top plot to lines & dots
//Dots()							Changes top plot to dots
//m(x)							Moves plot x-axis by x
//Ivt()							Sets labels 'current vs time'

//////////////FITTING
//CNTnoise						For fitting 1/f CNT noise in power spectral density. Optional fitting to a Lorentzian peak in addition to 1/f (associated with two state fluctuations).
//DoubleGauss					Two gaussian functions with shared widths.
////////////////////////////////////////////////////////////////////

//////////////////////////
//Color Coding Data
//////////////////////////

//Creates a number of waves to hold color coding information using the 'Color' function
Function MakeColors(wname)
	String wname
	Wave wavein=$wname

	Duplicate/O wavein, $(wname+"_red")/Wave=Red; DelayUpdate
	Duplicate/O wavein, $(wname+"_green")/Wave=Green; DelayUpdate
	Duplicate/O wavein, $(wname+"_blue")/Wave=Blue; DelayUpdate
	Duplicate/O wavein, $(wname+"_Orange")/Wave=Orange; DelayUpdate
	Duplicate/O wavein, $(wname+"_Black")/Wave=Black; DelayUpdate

	Red=NaN
	Green=NaN
	Blue=NaN
	Orange=NaN
	Black=NaN

	Display Red, Green, Blue, Orange, Black;DelayUpdate
//	ModifyGraph mode=2,rgb(Green)=(0,65280,0),rgb(Blue)=(0,15872,65280);DelayUpdate
//	ModifyGraph rgb(Orange)=(65280,43520,0),rgb(Black)=(0,0,0)
	ModifyGraph mode=2,rgb($(wname+"_green"))=(0,65280,0),rgb($(wname+"_blue"))=(0,15872,65280);DelayUpdate
	ModifyGraph rgb($(wname+"_orange"))=(65280,43520,0),rgb($(wname+"_black"))=(0,0,0)


End //MakeColors

//Copies data specified by cursors into a new wave
//For use color coding data
Function Color(wname, colorcode)
	String colorcode, 	wname
	Wave wavein=$wname
	Wave waveout=$(wname+"_"+colorcode)

	Variable i, min1, max1
	
	if(pcsr(A)<pcsr(B))
		min1=pcsr(A)
		max1=pcsr(B)
	else
		min1=pcsr(B)
		max1=pcsr(A)
	endif
	
	for(i=min1; i<max1; i+=1)	
		waveout[i]=wavein[i]	
	endfor
	
End //Color

//Deletes 'NaN' values, and 'stitches' together the numerical part of a wave. Useful following color coding in order to do further analysis.
Function Stitch(wname)
	String wname
	Wave wavein=$wname
	Variable i, j=-1
	
	Duplicate/O wavein, $(wname+"_stitch")/Wave=reddy; DelayUpdate
		
	for(i=dimsize(wavein,0); i>-1; i-=1)
		if(numtype(reddy[i]) == 2 && j==-1)
			j=i
		endif

		if(numtype(reddy[i]) != 2 && j!=-1)
			deletepoints i, j-i+1, reddy
			j=-1
		endif
		
		if(i==0 && j!=-1)
			deletepoints i, j-i+1, reddy
			j=-1
		endif		
		
	endfor
	
	display reddy;DelayUpdate
	ModifyGraph tick=2,mirror=1, axisOnTop(left)=1
	
End //Stitch

//Set's a section of a wave (as specified by the cursors) to NaN
Function DeleteMe(wavein)
	Wave wavein
	Variable i, min1, max1

	if(pcsr(A)<pcsr(B))
		min1=pcsr(A)
		max1=pcsr(B)
	else
		min1=pcsr(B)
		max1=pcsr(A)
	endif
	
	for(i=min1; i<max1; i+=1)	
		wavein[i]	= NaN
	endfor
			
End //DeleteMe

//////////////////////////
//Making audio files from ASCII data
//////////////////////////

//Creates a 16-bit wave (.wav) audio file from time domain data
Function MakeAudio(wavein)
	Wave wavein
	Variable rate = 1/deltax(wavein)  //Sample rate is in Hz
	Variable tarp, scalefactor, max1, min1, mean1, numpoints

	numpoints = dimsize(wavein,0)

	Make/W/O/N = (numpoints), newaudio

	tarp=1/rate

	SetScale/P x,0, tarp, newaudio
	
	min1=Abs(WaveMin(wavein))
	max1=Abs(WaveMax(wavein))
	if(min1>min1)
		scalefactor=30000*5/min1
	else
		scalefactor=30000*5/max1	
	endif

	mean1=mean(wavein)

	newaudio = (wavein-mean1)*scalefactor

End //MakeAudio

//////////////////////////
//Plotting functions
//////////////////////////

//Ploting using some formatting that I like
Function PlotMe(wavein)
	Wave wavein
	display wavein;DelayUpdate
	ModifyGraph tick=2,mirror=1,fSize=14,standoff=0;
End //Plotme

Function Lines()
	ModifyGraph mode=0
End //Lines

Function Linesanddots()
	ModifyGraph mode=4,marker=19,msize=1,rgb=(65280,0,0),useMrkStrokeRGB=1
End //Linesanddots

Function Dots()
	ModifyGraph mode=2,rgb=(0,0,0)
End //Dots

Function m(x) //Moves top axis by an amount specified by x
	Variable x

	getaxis bottom
	setaxis bottom V_min+x, V_max+x

End //m

Function Ivt()

	Label left "\\u#2current (nA)"; Label bottom "\\u#2time (sec)"; ModifyGraph tick=2,mirror=1,fSize=14,standoff=0;
	
end //Ivt

//////////////////////////
//Speeding the search for switching behavior
//////////////////////////

Function CorrelateMe(wname)
	String wname
	Wave wavein=$wname
	Wavestats wavein
	Duplicate/O wavein, $(wname+"_norm")/Wave=wave2; DelayUpdate
	wave2-=V_avg
	Duplicate/O wave2, $(wname+"_cor")/Wave=wave3; DelayUpdate
	Correlate/AUTO wave2, wave3
	display wave3;DelayUpdate
	ModifyGraph tick=2,mirror=1,fSize=14,standoff=0;
End //CorrelateMe

//Cut off a particular frequency range using a low pass filter
//and subtract the filtered data from the raw data
Function FilterMe(wname, cutoff, fouriercoeff)
	String wname
	Variable cutoff, fouriercoeff //fouriercoeff specifies the number of fourier coeficients to use in the low pass filter
	Wave wavein=$wname
	Variable f1, f2
	f1 = deltax(wavein)/cutoff	//Edge of pass band. Specified as a fraction of the sample rate
	f2=f1*1.02				//Edge of reject band. Set to 2% greater than edge of pass band

	Print "Removing timescales >", cutoff, "sec (freq <", f1/deltax(wavein), "Hz).", fouriercoeff, "Fourier terms used."

	//Plotting data
	display wavein;DelayUpdate
	ModifyGraph mode=2,rgb=(0,0,0), tick=2,mirror=1,fSize=14,standoff=0;
	//Doing the low pass filter
	Duplicate/O wavein, $(wname+"_smth")/Wave=wave2; DelayUpdate
	Make/O/D/N=0 $(wname+"_coefs")/Wave=wavecoef; DelayUpdate
	FilterFIR/DIM=0/LO={f1,f2,fouriercoeff}/COEF wavecoef, wave2

	//Plotting low pass filter result
	appendtograph wave2

	//Doing the subtraction
	Duplicate/O wavein,$(wname+"_sub")/Wave=wave3;DelayUpdate
	wave3=wavein-wave2
	
	//Plotting the date minus the boxcar average
	display wave3;DelayUpdate
	ModifyGraph mode=2,rgb=(0,0,0), tick=2,mirror=1,fSize=14,standoff=0;
		
End //FilterMe

Function PSDme(wname)
	String wname
	String cmd
	Variable scale //Holds a number for adjusting the x-axis scaling of the output PSD

	Wave wavein=$wname
	scale = 1/deltax(wavein)/65536		//The 65536 is the segmentation length of the PSD, if the segmentation length is adjusted this this value must be adjusted as well.
	print scale

	sprintf cmd, "BigPSD(\""+wname+"\",9,2,2)"		//preparing a string to call on the BigPSD macro
	print cmd
	Execute cmd		//Calling on the bigPSD macro

	Wave waveout=$(wname + "_psd")		//Making wave to extract some information from the output PSD
	print waveout[dimsize(waveout,0)]

	ModifyGraph log=1, mode=2,rgb=(0,0,0), tick=2,mirror=1, axisOnTop(left)=1, rgb=(0,0,0);DelayUpdate
	SetAxis left waveout[dimsize(waveout,0)]/2,waveout[2]*2	//Autoscales axis to something reasonable
	SetScale/P x 0,scale,"Hz", waveout
	
End //PSDme

Function BoxcarMe(wname, boxnum)
	String wname
	Variable boxnum
	Wave wavein=$wname

	//Plotting data
	display wavein;DelayUpdate
	ModifyGraph mode=2,rgb=(0,0,0)

	//Doing the boxcar averaging
	Duplicate/O wavein,$(wname+"_smth")/Wave=wave2;DelayUpdate
	Smooth/EVEN/B boxnum, wave2

	//Plotting boxcar average
	appendtograph wave2

	//Doing the subtraction
	Duplicate/O wavein,$(wname+"_sub")/Wave=wave3;DelayUpdate
	wave3=wavein-wave2
	
	//Plotting the date minus the boxcar average
	display wave3;DelayUpdate
	ModifyGraph mode=2,rgb=(0,0,0)
End //End BoxcarMe

//////////////////////////
//Fitting functions
//////////////////////////

Function CNTnoise(w,x) : FitFunc
	Wave w
	Variable x

	//CurveFitDialog/ These comments were created by the Curve Fitting dialog. Altering them will
	//CurveFitDialog/ make the function less convenient to work with in the Curve Fitting dialog.
	//CurveFitDialog/ Equation:
	//CurveFitDialog/ f(x) = A*x^j+B*G/(G^2+(2*pi*x)^2)
	//CurveFitDialog/ End of Equation
	//CurveFitDialog/ Independent Variables 1
	//CurveFitDialog/ x
	//CurveFitDialog/ Coefficients 4
	//CurveFitDialog/ w[0] = A
	//CurveFitDialog/ w[1] = j
	//CurveFitDialog/ w[2] = B
	//CurveFitDialog/ w[3] = G

	return w[0]*x^w[1]+w[2]*w[3]/(w[3]^2+(2*pi*x)^2)
End //CNTnoise

Function DoubleGauss(w,x) : FitFunc
	Wave w
	Variable x

	//CurveFitDialog/ These comments were created by the Curve Fitting dialog. Altering them will
	//CurveFitDialog/ make the function less convenient to work with in the Curve Fitting dialog.
	//CurveFitDialog/ Equation:
	//CurveFitDialog/ f(x) = A1/sqrt(3.14159/2)*exp(-2*(x-x1)^2/FWHM^2*ln(4))+A2/sqrt(3.14159/2)*exp(-2*(x-x2)^2/FWHM^2*ln(4))
	//CurveFitDialog/ End of Equation
	//CurveFitDialog/ Independent Variables 1
	//CurveFitDialog/ x
	//CurveFitDialog/ Coefficients 5
	//CurveFitDialog/ w[0] = A1
	//CurveFitDialog/ w[1] = x1
	//CurveFitDialog/ w[2] = A2
	//CurveFitDialog/ w[3] = x2
	//CurveFitDialog/ w[4] = FWHM

	return w[0]/sqrt(3.14159/2)*exp(-2*(x-w[1])^2/w[4]^2*ln(4))+w[2]/sqrt(3.14159/2)*exp(-2*(x-w[3])^2/w[4]^2*ln(4))
End

///////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////
//Asylum Research's functions for the BigPSD macro
///////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////

Constant Kb = 1.380662e-23;		//Boltzman constant in J / K

Function JustPink(w,x)			//a 1/f function
	wave w
	variable x
	
	return Sqrt(w[0]^2/x^w[1])

end //JustPink


macro BigPSD(w,seglen,window, Sqroot)					//an almost stock Igor provided maker of PSDs
	string w
	Prompt w "data wave:",popup WaveList("*",";","")
	variable seglen = 1
	Prompt seglen,"segment length:",popup "256;512;1024;2048;4096;8192;16384;32768;65536"
	variable window = 2
	Prompt window,"Window type:",popup "Square;Hann;Parzen;Welch;Hamming;"
						"BlackmanHarris3;KaiserBessel"
	variable Sqroot = 1
	Prompt Sqroot, "Pick units: ", popup "Linear (Units/Sqrt(Hz));Power (Units^2/Hz))"	

	BigPSDFunc(w,seglen,window,Sqroot)

	string destw = w+"_psd"
	
	if (!DoWindowOnList(destw,"Forward"))
		Display/K=1 $destw
//	BringDestFront(destw)
		if( numpnts($destw) <= 129 )
			ModifyGraph mode($destw)=4,marker($destw)=19,msize($destw)=1
			
		else
			ModifyGraph mode($destw) = 2
		endif
		ModifyGraph log(left) = 1
	endif
	
end //BigPSD

function BigPSDFunc(w,seglen,window,Sqroot)		//now a function with a macro front end
	string w
	variable seglen, window, Sqroot

variable start = StopMSTimer(-2)
	
	variable npsd = 2^(7+seglen)				// number of points in group (resultant psd wave len= npsd/2+1)
	variable psdOffset = npsd/2					// offset each group by this amount
	variable psdFirst = 0							// start of current group
	variable nsrc = numpnts($w)
	variable nsegs,winNorm						// count of number of segements and window normalization factor
	string destw = w+"_psd",srctmp = w+"_tmp"
	string winw = w+"_psdWin"					// window goes here
	
	if( npsd > nsrc/2 )
		Abort "psd: source wave should be MUCH longer than the segment length"
	endif
	make/O/N=(npsd/2+1) $destw
	make/O/N=(npsd) $srctmp,$winw
	wave WinWave = $winw
	wave SourceTemp = $srctmp
	wave DestWave = $destw
	WinWave = 1

	switch (window)
		case 1:
			winNorm = 1
			break
		case 2:
			Hanning WinWave
			winNorm = 0.372				//  winNorm is avg squared value
			break
		case 3:
			winNorm = Parzen(WinWave)
			break
		case 4:
			winNorm = Welch(WinWave)
			break
		case 5:
			winNorm = Hamming(WinWave)
			break
		case 6:
			winNorm = BlackmanHarris3(WinWave)
			break
		case 7:
			winNorm = KaiserBessel(WinWave)
			break
		default:
			Abort "unknown window index"
	endswitch
		
	Duplicate/O/R=[0,npsd-1] $w SourceTemp
	SourceTemp *= WinWave
	fft SourceTemp
	CopyScales/P SourceTemp, DestWave
	DestWave = magsqr(SourceTemp)
	psdFirst = psdOffset
	nsegs = 1
	do
		Duplicate/O/R=[psdFirst,psdFirst+npsd-1] $w SourceTemp
		SourceTemp *= WinWave
		fft SourceTemp
		DestWave += magsqr(SourceTemp)
		psdFirst += psdOffset
		nsegs += 1
	while (psdFirst+npsd < nsrc)
	winNorm = 2*deltax($w)/(winNorm*nsegs*npsd)
	DestWave *= winNorm
	DestWave[0] /= 2

	if (Sqroot == 1)
		DestWave = Sqrt(DestWave)
	endif
	KillWaves SourceTemp, WinWave

print (StopMSTimer(-2)-start)/1e6

end //BigPSDFunc

Function DoWindowOnList(waveListStr,action)		//do the action on windows that have the waves in the list on them
	string waveListStr, action
	
	string windowList = WaveInWindowsList(waveListStr)	//get a list of all of the windows with the waves on them
	variable i, j, iStop, jStop
	
	if (strlen(windowList) == 0)			//nothing in the list, return 0 for zero windows
		return 0
	endif
	
	strswitch (action)						//do the action
		
		case "Forward":
			DoWindow/F $StringFromList(0,windowList)		//just bring the first window forward
			break
			
		case "Kill":							//kill all the windows
			
			iStop = ItemsInList(windowList)		//set the stop
			for (i = 0;i < iStop;i += 1)
				DoWindow/K $StringFromList(i,windowList)	//kill them all
			endfor
			break

		case "Remove":						//remove all of the traces
			
			iStop = ItemsInList(windowList)		//stop for the window list
			jStop = ItemsInList(waveListStr)		//stop for the wave list
			for (i = 0;i < iStop;i += 1)
				for (j = 0;j < jStop;j += 1)
					RemoveFromGraph/Z/W=$StringFromList(i,windowList) $StringFromList(j,waveListStr)	//remove if there
					RemoveImage/Z/W=$StringFromList(i,windowList) $StringFromList(j,waveListStr)			//remove if there
					if (FindListItem(StringFromList(j,waveListStr),ContourNameList(StringFromList(j,windowList),";"),";",0,0) >= 0)
						RemoveContour/W=$StringFromList(i,windowList) $StringFromList(j,waveListStr)		//no /Z flag, so you have to check first
					endif
				endfor
			endfor
			break
			
		default:				//the requested action was not recognized
			print "That action is not one of the blessed actions"
		
	endswitch
	
	return ItemsInList(windowList)
	
end //DoWindowOnList

Function/S WaveInWindowsList(waveListStr)	//this function returns a list of the windows that the waves it is passed are in
	string waveListStr							//a list of waves
	
	string windowList = WinList("*",";","WIN:3")		//find all graphs and tables
	string currentStr
	string resultList = ""
	variable i, j
	
	for (j = 0;j < ItemsInList(windowList);j += 1)			//go through the window list, check for images, contours, and traces

		for (i = 0;i < ItemsInList(waveListStr);i += 1)	//go through the wave list

			currentStr = StringFromList(i,waveListStr)

			if (FindListItem(currentStr,TraceNameList(StringFromList(j,windowList),";",1),";",0,0) >= 0)		//the or statement was just too long
				resultList = Add2StrList(resultList,StringFromList(j,windowList))		//append on if the wave was on the window
			elseif (FindListItem(currentStr,ImageNameList(StringFromList(j,windowList),";"),";",0,0) >= 0)
				resultList = Add2StrList(resultList,StringFromList(j,windowList))		//append on if the wave was on the window
			elseif (FindListItem(currentStr,ContourNameList(StringFromList(j,windowList),";"),";",0,0) >= 0)
				resultList = Add2StrList(resultList,StringFromList(j,windowList))		//append on if the wave was on the window
			endif
		endfor
		
	endfor
	
	return resultList		//return the list
	
End //WaveInWindowsList

Function/S Add2StrList(StrList,Str,[Sep])
	String StrList, Str
	String Sep
	if (ParamIsDefault(Sep))
		Sep = ";"
	endif
	
//	Variable ind = WhichListItem(lowerStr(Str),LowerStr(StrList),Sep,0)
	Variable ind = WhichListItem(Str,StrList,Sep,0,0)
	if (Ind >= 0)
		return(StrList)
	endif
	if (Strlen(StrList) && (cmpstr(StrList[strlen(StrList)-1],Sep) != 0))
		Str = Sep+Str		//has the effect of:
		//StrList += Sep
		//but is faster, because Str is shorter.
	endif
//	if (ind == -1)
//		ARC_StrCat(StrList,Str+Sep)
		StrList += Str+Sep
//		Strlist += Str+Sep
//	endif
	return(Strlist)
End //Add2StrList

///////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////
//End of Asylum Research's functions for the BigPSD macro
///////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////